/*=============================================================
  ===    EDITIONS TRANSOCEANIC  -  ELECTRONIQUE PRATIQUE    ===
  ===              (c) Yves MERGY 11/2010                   ===
  =============================================================
  ===      PROGRAMME POUR LE ROBOT EXPERIMENTAL ARDUINO     ===
  ===            FONCTIONNEMENT SELON 3 MODES :             ===
  ===  *- DEPLACEMENTS LIBRES AVEC DETECTION DES OBSTACLES  ===
  ===  *- A COMMANDE PAR L'ACCELEROMETRE DU "NUNCHUCK"      ===
  ===  *- A COMMANDE PAR LE JOYSTICK DU "NUNCHUCK"          ===
  ===         GESTION DE DIFFERENTS PERIPHERIQUES :         ===
  ===  *- UN AFFICHEUR LCD DE 4 x 20 C                      ===
  ===  *- 2 MOTEURS COMMANDES EN "PWM" ou "MLI"             ===
  ===  *- UNE MANETTE DE JEUX "WII NUNCHUCK"                ===
  =============================================================

CONFIGURATION DU PORT I2C POUR LA MANETTE DE JEUX "WII NUNCHUCK"
    SDA sur port P18 (ANA4
    SCK sur port P19 (ANA5)
  Lecture des valeurs mini et maxi de la maquette:
    Accélérateur X: mini=73 maxi=183 ==> de Gauche vers Droite Repos = 135
    Accélérateur Y: mini=76 maxi=184 ==> de l'Arrière vers l'avant Repos = 135
    Accélérateur Z: mini=77 maxi=191 ==> du Dessous vers le Dessus Repos = 135
    Joystick X: mini=25 maxi=223 ==> de Gauche vers Droite Repos = 125
    Joystick Y: mini=30 maxi=225 ==> de l'Arrière vers l'avant Repos = 125
    Bouton C=1 au repos C=0 appuyé
    Bouton Z=1 au repos Z=0 appuyé

CONFIGURATION DE L'AFFICHEUR LCD
    Gestion en mode 4 bits et RW non utilisé
    RS sour port 17 (ANA3)
    EN sur port 16 (ANA2)
    D4 à D7 sur ports D10 à D13 (D0 à D3 non utilisées)
*/
//============================== AJOUT DE LIBRAIRIES ADDITIONNELLES
#include <LiquidCrystal.h>  // Librairie pour l'afficheur alphanumérique LCD
#include <Wire.h>           // Librairie pour la communication I2C (pour le nunchuck)
  
//============================== DECLARATION DE LA VARIABLE "LCD" DE TYPE "LiquidCrystal"
  // Mode 4 bits avec RW non connecté (mode le plus simple)
  // Configuration: RS=17 - EN=16 - D4=10 - D5=11 - D6=12 - D7=13 - RW=Masse
  LiquidCrystal LCD(17, 16, 10, 11, 12, 13);
  
//============================== DECLARATION DES VARIABLES ATTRIBUEES AUX BROCHES D'E/S
  int MDV = 3;  // Vitesse du moteur droit sur Port 3 (PWM utilisé)
  int MD1 = 2;  // Commande 1 du moteur droit sur Port 2
  int MD2 = 4;  // Commande 2 du moteur droit sur Port 4
  int MGV = 9;  // Vitesse du moteur gauche sur Port 9 (PWM utilisé)
  int MG1 = 5;  // Commande 1 du moteur gauche sur Port 5 (PWM)
  int MG2 = 6;  // Commande 2 du moteur gauche sur Port 6 (PWM)
  int BPR = 7;  // Touche rouge sur Port 7
  int BPB = 8;  // Touche bleue sur Port 8
  int PCD = 14; // Touche rouge sur Port 14 (ANA0)
  int PCG = 15; // Touche bleue sur Port 15 (ANA1)
  
//============================== DECLARATION DES VARIABLES DU PROGRAMME
  unsigned int VITG;     // variable de vitesse pour le moteur Gauche
  unsigned int VITD;     // variable de vitesse pour le moteur Droit
  int DIR;               // variable de direction
  byte Tableau_WII[6];   // Tableau pour stocker les 7 valeurs du NUNCHUCK au format uint8_t ou byte
  int Bouton_Z;          // variable de vitesse pour l'état logique du bouton Z du nunchuck
  int Bouton_C;          // variable de vitesse pour l'état logique du bouton C du nunchuck
  int Joy_X;             // variable de la position en X du joystick du nunchuck
  int Joy_Y;             // variable de la position en Y du joystick du nunchuck
  int Acc_X;             // variable de la position en X de l'accéléromètre du nunchuck
  int Acc_Y;             // variable de la position en Y de l'accéléromètre du nunchuck
  int Acc_Z;             // variable de la position en Z de l'accéléromètre du nunchuck
  byte Mode;             // variable du Mode de déplacement sélectionné
  byte Touche;           // variable de l'état logique de la touche
  byte Affiche;          // Variable drapeau pour affichage ou non

//============================== PROCEDURE D'INITIALISATION OBLIGATOIRE NOMMEE "setup"
void setup() {
  //Serial.begin(19200);               // Supprimé. Utile uniquement lors de la mise au point
  //Serial.print ("Terminal prêt\n");  // Supprimé. Utile uniquement lors de la mise au point
  I2C_init();                          // Appel de la procédure d'initialisation I2C pour le Nunchuck

  pinMode(MDV, OUTPUT);   // Configuration des broches d'E/S
  pinMode(MD1, OUTPUT);
  pinMode(MD2, OUTPUT);
  pinMode(MGV, OUTPUT);
  pinMode(MG1, OUTPUT);
  pinMode(MG2, OUTPUT);
  pinMode(BPR, INPUT);
  pinMode(BPB, INPUT);
  pinMode(PCD, INPUT);
  pinMode(PCG, INPUT);
  digitalWrite(MDV, LOW);
  digitalWrite(MD1, LOW);
  digitalWrite(MD2, LOW);
  digitalWrite(MGV, LOW);
  digitalWrite(MG1, LOW);
  digitalWrite(MG2, LOW); //........

// Préparation de l'afficheur LCD et affichage du message
  LCD.begin(20,4); // Initialise le LCD avec 20 colonnes x 4 lignes 
  delay(50);       // pause rapide pour laisser temps initialisation
  LCD.setCursor(0, 0) ;              // Positionne le curseur colonne 0; ligne 0
  LCD.print("EDITION TRANSOCEANIC"); // Affiche le message de présentation sur 4 lignes
  LCD.setCursor(1, 1) ;              // Positionne le curseur colonne 1; ligne 1
  LCD.print("ROBOT EXPERIMENTAL");   // Affiche le message de présentation
  LCD.setCursor(6, 2) ;              // Positionne le curseur colonne 6; ligne 2
  LCD.print("ARDUINO");              // Affiche le message de présentation
  LCD.setCursor(4, 3) ;              // Positionne le curseur colonne 4; ligne 3
  LCD.print("WII NUNCHUCK");         // Affiche le message de présentation
  VITG = 60;    // Valeur PWM déterminant le rapport cyclique pour la vitesse du moteur Gauche
  VITD = 60;    // Valeur PWM déterminant le rapport cyclique pour la vitesse du moteur Droit
  Mode = 0;     // Mode 0 par défaut au départ (appelle la procédure de choix)
  Affiche = 0;  // Affichage au départ du programme
  delay(1000);  // Temporisation d'une seconde pour lire le message de présentation
}

//============================== BOUCLE PRINCIPALE OBLIGATOIRE NOMMEE "loop"
void loop() {
  Touche = digitalRead(BPB);                // Lecture de la touche Bleue
  if (Touche == LOW && Mode != 1) Choix();  // Procédure de choix si actionnée et Mode différent de 1
  if (Mode == 0) Choix();                   // Choix si Mode = 0
  if (Mode == 1) ROULE_LIB();               // Déplacements libres si Mode = 1
  if (Mode == 2) ROULE_ACC();               // Déplacements par l'accéléromètre si Mode = 2
  if (Mode == 3) ROULE_JOY();               // Déplacements par le joystick si Mode = 3
  delay(100);                        // Temporisation d'1/10 ème de S. pour ralentir de déroulement
}

//============================== FONCTIONS SUPPLEMENTAIRES UTILISATEUR =================

//========== FONCTION DE CHOIX DU MODE DE FONCTIONNEMENT ===============================
void Choix() {       // Début de procédure
  if (digitalRead(BPB) == LOW) Mode++; // Mode incrémenté si touche Bleue actionnée
  do {                                 // Antirebonds : bouclage tant que la touche
    Touche = digitalRead(BPB);         // .........
  } while (Touche == LOW);             // ... est actionnée.
  delay (100);
  if (Mode >= 4) Mode = 1;             // Test si dépassement de valeur du mode
  LCD.setCursor(0, 0) ;                // Positionnement du curseur (Colonne 0 - Ligne 0)
  if (Mode == 0) LCD.print("CHOISISSEZ UN MODE. "); // Affichage sur la 1ère ligne ...
  if (Mode == 1) LCD.print("DEPLACEMENTS LIBRES "); // ... en fonction de la valeur ...
  if (Mode == 2) LCD.print("PAR L'ACCELEROMETRE "); // ... du mode ...
  if (Mode == 3) LCD.print("PAR LE JOYSTICK     "); // ....
  LCD.setCursor(0, 1);                 // Positionnement du curseur (Colonne 0 - Ligne 1)
  LCD.print("====================");   // Affichage
  LCD.setCursor(0, 2);                 // Positionnement du curseur (Colonne 0 - Ligne 2)
  LCD.print("[BLEUE]-> CHOIX     ");   // Affichage
  LCD.setCursor(0, 3);                 // Positionnement du curseur (Colonne 0 - Ligne 3)
  LCD.print("[ROUGE]-> VALIDATION");   // Affichage
  if (digitalRead(BPR) == HIGH) Choix();  // Choix si touche Rouge actionnée
  do {                                    // Antirebonds : bouclage tant que la touche
    Touche = digitalRead(BPR);            // .........
  } while (Touche == LOW);                // ... est actionnée.
}                  // Fin de procédure


//========== FONCTION D'INITIALISATION DU NUNCHUNCK ==================================
  // Initialise le port I2C
  // et le prépare à communiquer avec le Nunchunck
void I2C_init()
{ 
  Wire.begin();	                // Relier le bus I2C comme maître
  Wire.beginTransmission(0x52);	// Transmet à l'adresse de base du Nunchunck 0x52
  Wire.send(0x40);		// Envoie l'adresse mémoire
  Wire.send(0x00);		// Envoie un zéro  
  Wire.endTransmission();	// Arrêt de transmission
}

//========== FONCTION DE DEMANDE DE DONNEES DU NUNCHUCK ===============================
void I2C_demande() {
  Wire.beginTransmission(0x52);	// Transmet à l'adresse de base du Nunchunck 0x52
  Wire.send(0x00);		// Envoie un octet
  Wire.endTransmission();	// Arrêt de transmission
}

//========== FONCTION DE LECTURE DES DONNEES DU NUNCHUCK ===============================
int I2C_lecture() {
    int COMPTEUR = 0;
    Wire.requestFrom (0x52, 6);	// Demande de données provenant du nunchuck
    while (Wire.available ()) {
      // Reçoit un octet (byte) comme un entier (Integer)
      Tableau_WII[COMPTEUR] = Format_WII(Wire.receive());
      COMPTEUR ++;
    }
    I2C_demande();  // Envoi d'une demande de données
    // Si les 6 octets sont reçus, ils sont affichés
    if (COMPTEUR >= 5) return 1;   //succès
    return 0;    //erreur
}

//========== FONCTION DE CALCULS ET D'AFFICHAGE DES DONNEES RECUES =============================
void I2C_calculs() { 
  static int i=0;
  Joy_X = Tableau_WII[0];
  Joy_Y = Tableau_WII[1];
  Acc_X = Tableau_WII[2]; 
  Acc_Y = Tableau_WII[3];
  Acc_Z = Tableau_WII[4];
  Bouton_Z = 0;
  Bouton_C = 0;
  // L'octet "Tableau_WII[5]" contient les bits pour les boutons Z et C
  // ainsi que les bits de poids faibles des données de l'accéléromètre,
  // il est donc nécessaire de tester chaque bit de cet octet.
  // Les valeurs de l'accéléromètre sont sur 10 bits
  // Il faut lire l'octet correspondant sur 8 bits dans le tableau, et additionner
  // les 2 derniers bits lus dans l'octet "Tableau_WII[5]".
  if ((Tableau_WII[5] >> 0) & 1) Bouton_Z = 1;
  if ((Tableau_WII[5] >> 1) & 1) Bouton_C = 1;
  if ((Tableau_WII[5] >> 2) & 1) Acc_X += 2;
  if ((Tableau_WII[5] >> 3) & 1) Acc_X += 1;
  if ((Tableau_WII[5] >> 4) & 1) Acc_Y += 2;
  if ((Tableau_WII[5] >> 5) & 1) Acc_Y += 1;
  if ((Tableau_WII[5] >> 6) & 1) Acc_Z += 2;
  if ((Tableau_WII[5] >> 7) & 1) Acc_Z += 1;
  // Affichage des données reçues.
  LCD.setCursor(0, 0) ;              // Positionne le curseur colonne 0; ligne 0
  LCD.print("Lecture No:         ");
  LCD.setCursor(0, 1) ;              // Positionne le curseur colonne 0; ligne 1
  LCD.print("Joystick X:    Y:   ");
  LCD.setCursor(0, 2) ;              // Positionne le curseur colonne 0; ligne 2
  LCD.print("Ac X:    Y:    Z:   ");
  LCD.setCursor(0, 3) ;              // Positionne le curseur colonne 0; ligne 3
  LCD.print("Boutons C:    Z:    ");
  LCD.setCursor(11, 0) ;              // Positionne le curseur colonne 11; ligne 0
  LCD.print(i,DEC);
  LCD.setCursor(11, 1) ;              // Positionne le curseur colonne 11; ligne 1
  LCD.print(Joy_X,DEC);
  LCD.setCursor(17, 1) ;              // Positionne le curseur colonne 17; ligne 1
  LCD.print(Joy_Y,DEC);
  LCD.setCursor(5, 2) ;              // Positionne le curseur colonne 5; ligne 2
  LCD.print(Acc_X,DEC);
  LCD.setCursor(11, 2) ;              // Positionne le curseur colonne 11; ligne 2
  LCD.print(Acc_Y,DEC);
  LCD.setCursor(17, 2) ;              // Positionne le curseur colonne 17; ligne 2
  LCD.print(Acc_Z,DEC);
  LCD.setCursor(10, 3) ;              // Positionne le curseur colonne 10; ligne 3
  LCD.print(Bouton_C,DEC);
  LCD.setCursor(16, 3) ;              // Positionne le curseur colonne 16; ligne 3
  LCD.print(Bouton_Z,DEC);
  i++;
}

//========== FONCTION D'ENCODAGE DES DONNEES AU FORMAT DE LA WII ===============================
char Format_WII (char x) {
  x = (x ^ 0x17) + 0x17;
  return x;
}


//========== FONCTION DE DEPLACEMENTS LIBRES ===============================
void ROULE_LIB(){
  VITG = 100;
  VITD = 100;
  if (Affiche == 0)
  {
    LCD.setCursor(0, 0) ;              // Positionne le curseur colonne 0; ligne 0
    LCD.print("MODE DE DEPLACEMENT ");
    LCD.setCursor(0, 1) ;              // Positionne le curseur colonne 0; ligne 1
    LCD.print("LIBRE AVEC DETECTION");
    LCD.setCursor(0, 2) ;              // Positionne le curseur colonne 0; ligne 2
    LCD.print("[0]<G-OBSTACLE-D>[0]");
    LCD.setCursor(0, 3) ;              // Positionne le curseur colonne 0; ligne 3
    LCD.print("FONCTION: AVANT     ");
    Affiche = 1;
  }
  LCD.setCursor(10, 3) ;
  LCD.print("AVANT     ");
  Avant();
  if (digitalRead(PCD) == LOW)
  {
    LCD.setCursor(18, 2) ;
    LCD.print("1");
    LCD.setCursor(10, 3) ;
    LCD.print("ARRIERE   ");
    Arriere();
    delay (600);
    LCD.setCursor(10, 3) ;
    LCD.print("STOP      ");
    Arret();
    delay (500);
    LCD.setCursor(10, 3) ;
    LCD.print("GAUCHE    ");
    Virage_G();
    delay (400);
    LCD.setCursor(10, 3) ;
    LCD.print("STOP      ");
    Arret();
    delay (500);
    LCD.setCursor(18, 2) ;
    LCD.print("0");
  }
  if (digitalRead(PCG) == LOW)
  {
    LCD.setCursor(1, 2) ;
    LCD.print("1");
    LCD.setCursor(10, 3) ;
    LCD.print("ARRIERE   ");
    Arriere();
    delay (600);
    LCD.setCursor(10, 3) ;
    LCD.print("STOP      ");
    Arret();
    delay (500);
    LCD.setCursor(10, 3) ;
    LCD.print("DROITE    ");
    Virage_D();
    delay (400);
    LCD.setCursor(10, 3) ;
    LCD.print("STOP      ");
    Arret();
    delay (500);
    LCD.setCursor(1, 2) ;
    LCD.print("0");
  }
  if (digitalRead (BPR) == LOW)
  {
    Affiche = 0;
    Arret();
    Mode = 0;
    Choix();
  }
}

//========== FONCTION DE DEPLACEMENTS A L'AIDE DES ACCELEROMETRES ===============================
void ROULE_ACC(){
  I2C_lecture();
  I2C_calculs();
// Si le bouton Z est relaché, le robot s'arrête.
  if (Bouton_Z == 1)
  {
    Arret();
  }
  else  // Sinon ...
  {
    //... Arrêt si l'accéléromètre en Y est au centre
    if (Acc_Y >=120 && Acc_Y <=140)
    {
      VITG = 0;
      VITD = 0;
      Arret();
    }
    //... Avant si l'accéléromètre en Y (nunchuck) est vers l'avant
    // avec vitesse progressive en fonction de son inclinaison.
    if (Acc_Y > 140) 
    {
      VITG = (Acc_Y - 140)*5;
      if (VITG >=210) VITG = 255;
      VITD = VITG;
      if (Acc_X < 110) VITG = VITD / 2;
      if (Acc_X > 150) VITD = VITG / 2;
      Avant();
    }
    //... Arrière si l'accéléromètre en Y (nunchuck) est vers l'arrière
    // avec vitesse progressive en fonction de son inclinaison.
    if (Acc_Y < 120) 
    {
      VITG = (120 - Acc_Y)*5;
      if (VITG >=210) VITG = 255;
      VITD = VITG;
      if (Acc_X < 110) VITG = VITD / 2;
      if (Acc_X > 150) VITD = VITG / 2;
      Arriere();
    }
  }
}

//========== FONCTION DE DEPLACEMENTS A L'AIDE DU JOYSYTICK ===============================
void ROULE_JOY(){
  I2C_lecture();
  I2C_calculs();
  //... Arrêt si le joystick est au centre
  if (Joy_Y >=115 && Joy_Y <=135)
  {
    VITG = 0;
    VITD = 0;
    Arret();
  }
  //... Avant si le joystick en Y est vers l'avant
  // avec vitesse progressive en fonction de la position.
  if (Joy_Y > 135) 
  {
    VITG = (Joy_Y - 135)*3;
    if (VITG >=255) VITG = 255;
    VITD = VITG;
    if (Joy_X < 115) VITG = VITD / 2;
    if (Joy_X > 135) VITD = VITG / 2;
    Avant();
  }
  //... Arrière si l'accéléromètre en Y (nunchuck) est vers l'arrière
  // avec vitesse progressive en fonction de son inclinaison.
  if (Joy_Y < 115) 
  {
    VITG = (115 - Joy_Y)*3;
    if (VITG >=255) VITG = 255;
    VITD = VITG;
    if (Joy_X < 115) VITG = VITD / 2;
    if (Joy_X > 135) VITD = VITG / 2;
    Arriere();
  }
}

//========== FONCTION D'ARRET ===============================
void Arret(){
  //Serial.print ("STOP\n");
  digitalWrite(MG1, LOW);
  digitalWrite(MG2, LOW);
  analogWrite(MGV, 0);
  digitalWrite(MD1, LOW);
  digitalWrite(MD2, LOW);
  analogWrite(MDV, 0);
  delay(10);
}

//========== FONCTION DE MARCHE ARRIERE ===============================
void Arriere(){
  //Serial.print ("Arrière\n");
  digitalWrite(MG1, HIGH);
  digitalWrite(MG2, LOW);
  analogWrite(MGV, VITG);
  digitalWrite(MD1, LOW);
  digitalWrite(MD2, HIGH);
  analogWrite(MDV, VITD);
}

//========== FONCTION DE MARCHE AVANT ===============================
void Avant(){
  //Serial.print ("Avant\n");
  digitalWrite(MG1, LOW);
  digitalWrite(MG2, HIGH);
  analogWrite(MGV, VITG);
  digitalWrite(MD1, HIGH);
  digitalWrite(MD2, LOW);
  analogWrite(MDV, VITD);
}

//========== FONCTION DE VIRAGE A DROITE ===============================
void Virage_D(){
  //Serial.print ("Gauche\n");
  digitalWrite(MG1, LOW);
  digitalWrite(MG2, HIGH);
  analogWrite(MGV, VITG);
  digitalWrite(MD1, LOW);
  digitalWrite(MD2, HIGH);
  analogWrite(MDV, VITD);
}

//========== FONCTION DE VIRAGE A GAUCHE ===============================
void Virage_G(){
  //Serial.print ("Droite\n");
  digitalWrite(MG1, HIGH);
  digitalWrite(MG2, LOW);
  analogWrite(MGV, VITG);
  digitalWrite(MD1, HIGH);
  digitalWrite(MD2, LOW);
  analogWrite(MDV, VITD);
}

