/************************************************************************
 *
 *  Module:       usbio_i.h
 *  Long name:    USBIO Driver Interface
 *  Description:  Defines the programming interface (API) of the USBIO driver
 *
 *  Runtime Env.: Win32
 *  Author(s):    Guenter Hildebrandt, Udo Eberhardt
 *  Company:      Thesycon GmbH, Ilmenau
 ************************************************************************/

#ifndef _USBIO_I_H_
#define _USBIO_I_H_

//
// Define the API version number.
// This will be incremented if changes are made.
//
// Applications should check if the driver supports the 
// required API version using IOCTL_USBIO_GET_DRIVER_INFO.
//
// current API version: 1.41
#define USBIO_API_VERSION   0x0141

//
// built-in (default) GUID for the USBIO interface
//
// IMPORTANT:
// Don't use this GUID for your own (custom) application. Generate a new one
// that is private to you and put it in the USBIO INF file and use it in 
// your application. Refer to the USBIO manual for further instructions.
// The tool guidgen.exe from the Microsoft Platform SDK can be used to create private GUIDs.
//
#define USBIO_IID_STR     "{325ddf96-938c-11d3-9e34-0080c82727f4}"
#define USBIO_IID_STR_W  L"{325ddf96-938c-11d3-9e34-0080c82727f4}"

#define USBIO_IID {0x325ddf96, 0x938c, 0x11d3, {0x9e, 0x34, 0x00, 0x80, 0xc8, 0x27, 0x27, 0xf4}}


//
// USBIO Error Codes
//
#define USBIO_ERR_SUCCESS                  0x00000000L
#define USBIO_ERR_CRC                      0xE0000001L
#define USBIO_ERR_BTSTUFF                  0xE0000002L
#define USBIO_ERR_DATA_TOGGLE_MISMATCH     0xE0000003L
#define USBIO_ERR_STALL_PID                0xE0000004L
#define USBIO_ERR_DEV_NOT_RESPONDING       0xE0000005L
#define USBIO_ERR_PID_CHECK_FAILURE        0xE0000006L
#define USBIO_ERR_UNEXPECTED_PID           0xE0000007L
#define USBIO_ERR_DATA_OVERRUN             0xE0000008L
#define USBIO_ERR_DATA_UNDERRUN            0xE0000009L
#define USBIO_ERR_RESERVED1                0xE000000AL
#define USBIO_ERR_RESERVED2                0xE000000BL
#define USBIO_ERR_BUFFER_OVERRUN           0xE000000CL
#define USBIO_ERR_BUFFER_UNDERRUN          0xE000000DL
#define USBIO_ERR_NOT_ACCESSED             0xE000000FL
#define USBIO_ERR_FIFO                     0xE0000010L
#define USBIO_ERR_ENDPOINT_HALTED          0xE0000030L
#define USBIO_ERR_NO_MEMORY                0xE0000100L
#define USBIO_ERR_INVALID_URB_FUNCTION     0xE0000200L
#define USBIO_ERR_INVALID_PARAMETER        0xE0000300L
#define USBIO_ERR_ERROR_BUSY               0xE0000400L
#define USBIO_ERR_REQUEST_FAILED           0xE0000500L
#define USBIO_ERR_INVALID_PIPE_HANDLE      0xE0000600L
#define USBIO_ERR_NO_BANDWIDTH             0xE0000700L
#define USBIO_ERR_INTERNAL_HC_ERROR        0xE0000800L
#define USBIO_ERR_ERROR_SHORT_TRANSFER     0xE0000900L
#define USBIO_ERR_BAD_START_FRAME          0xE0000A00L
#define USBIO_ERR_ISOCH_REQUEST_FAILED     0xE0000B00L
#define USBIO_ERR_FRAME_CONTROL_OWNED      0xE0000C00L
#define USBIO_ERR_FRAME_CONTROL_NOT_OWNED  0xE0000D00L

#define USBIO_ERR_CANCELED                 0xE0010000L
#define USBIO_ERR_CANCELING                0xE0020000L

#define USBIO_ERR_FAILED                   0xE0001000L
#define USBIO_ERR_INVALID_INBUFFER         0xE0001001L
#define USBIO_ERR_INVALID_OUTBUFFER        0xE0001002L
#define USBIO_ERR_OUT_OF_MEMORY            0xE0001003L
#define USBIO_ERR_PENDING_REQUESTS         0xE0001004L
#define USBIO_ERR_ALREADY_CONFIGURED       0xE0001005L
#define USBIO_ERR_NOT_CONFIGURED           0xE0001006L
#define USBIO_ERR_OPEN_PIPES               0xE0001007L
#define USBIO_ERR_ALREADY_BOUND            0xE0001008L
#define USBIO_ERR_NOT_BOUND                0xE0001009L
#define USBIO_ERR_DEVICE_NOT_PRESENT       0xE000100AL
#define USBIO_ERR_CONTROL_NOT_SUPPORTED    0xE000100BL
#define USBIO_ERR_TIMEOUT                  0xE000100CL
#define USBIO_ERR_INVALID_RECIPIENT        0xE000100DL
#define USBIO_ERR_INVALID_TYPE             0xE000100EL
#define USBIO_ERR_INVALID_IOCTL            0xE000100FL
#define USBIO_ERR_INVALID_DIRECTION        0xE0001010L
#define USBIO_ERR_TOO_MUCH_ISO_PACKETS     0xE0001011L
#define USBIO_ERR_POOL_EMPTY               0xE0001012L
#define USBIO_ERR_PIPE_NOT_FOUND           0xE0001013L
#define USBIO_ERR_INVALID_ISO_PACKET       0xE0001014L
#define USBIO_ERR_OUT_OF_ADDRESS_SPACE     0xE0001015L
#define USBIO_ERR_INTERFACE_NOT_FOUND      0xE0001016L
#define USBIO_ERR_INVALID_DEVICE_STATE     0xE0001017L
#define USBIO_ERR_INVALID_PARAM            0xE0001018L
#define USBIO_ERR_DEMO_EXPIRED             0xE0001019L
#define USBIO_ERR_INVALID_POWER_STATE      0xE000101AL
#define USBIO_ERR_POWER_DOWN               0xE000101BL
#define USBIO_ERR_VERSION_MISMATCH         0xE000101CL
#define USBIO_ERR_SET_CONFIGURATION_FAILED 0xE000101DL

#define USBIO_ERR_VID_RESTRICTION          0xE0001080L
#define USBIO_ERR_ISO_RESTRICTION          0xE0001081L
#define USBIO_ERR_BULK_RESTRICTION         0xE0001082L
#define USBIO_ERR_EP0_RESTRICTION          0xE0001083L
#define USBIO_ERR_PIPE_RESTRICTION         0xE0001084L
#define USBIO_ERR_PIPE_SIZE_RESTRICTION    0xE0001085L

#define USBIO_ERR_DEVICE_NOT_FOUND         0xE0001100L
#define USBIO_ERR_DEVICE_NOT_OPEN          0xE0001102L
#define USBIO_ERR_NO_SUCH_DEVICE_INSTANCE  0xE0001104L
#define USBIO_ERR_INVALID_FUNCTION_PARAM   0xE0001105L


//
// USB Standard Device Requests
//

#define IOCTL_USBIO_GET_DESCRIPTOR            _USBIO_IOCTL_CODE(1,METHOD_OUT_DIRECT)
// InBuffer:  USBIO_DESCRIPTOR_REQUEST
// OutBuffer: descriptor data, variable size!

#define IOCTL_USBIO_SET_DESCRIPTOR            _USBIO_IOCTL_CODE(2,METHOD_IN_DIRECT)
// InBuffer:  USBIO_DESCRIPTOR_REQUEST
// OutBuffer: descriptor data, variable size!



#define IOCTL_USBIO_SET_FEATURE               _USBIO_IOCTL_CODE(3,METHOD_BUFFERED)
// InBuffer:  USBIO_FEATURE_REQUEST
// OutBuffer: none

#define IOCTL_USBIO_CLEAR_FEATURE             _USBIO_IOCTL_CODE(4,METHOD_BUFFERED)
// InBuffer:  USBIO_FEATURE_REQUEST
// OutBuffer: none



#define IOCTL_USBIO_GET_STATUS                _USBIO_IOCTL_CODE(5,METHOD_BUFFERED)
// InBuffer:  USBIO_STATUS_REQUEST
// OutBuffer: USBIO_STATUS_REQUEST_DATA



#define IOCTL_USBIO_GET_CONFIGURATION         _USBIO_IOCTL_CODE(6,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: USBIO_GET_CONFIGURATION_DATA


#define IOCTL_USBIO_GET_INTERFACE             _USBIO_IOCTL_CODE(7,METHOD_BUFFERED)
// InBuffer:  USBIO_GET_INTERFACE
// OutBuffer: USBIO_GET_INTERFACE_DATA


#define IOCTL_USBIO_STORE_CONFIG_DESCRIPTOR   _USBIO_IOCTL_CODE(8,METHOD_BUFFERED)
// InBuffer:  USB_CONFIGURATION_DESCRIPTOR variable size!
// OutBuffer: none

#define IOCTL_USBIO_SET_CONFIGURATION         _USBIO_IOCTL_CODE(9,METHOD_BUFFERED)
// InBuffer:  USBIO_SET_CONFIGURATION
// OutBuffer: none

#define IOCTL_USBIO_UNCONFIGURE_DEVICE        _USBIO_IOCTL_CODE(10,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: none

#define IOCTL_USBIO_SET_INTERFACE             _USBIO_IOCTL_CODE(11,METHOD_BUFFERED)
// InBuffer:  USBIO_INTERFACE_SETTING
// OutBuffer: none



//
// USB Class or Vendor specific Device Requests
//

#define IOCTL_USBIO_CLASS_OR_VENDOR_IN_REQUEST  _USBIO_IOCTL_CODE(12,METHOD_OUT_DIRECT)
// InBuffer:  USBIO_CLASS_OR_VENDOR_REQUEST
// OutBuffer: data, variable size!

#define IOCTL_USBIO_CLASS_OR_VENDOR_OUT_REQUEST _USBIO_IOCTL_CODE(13,METHOD_IN_DIRECT)
// InBuffer:  USBIO_CLASS_OR_VENDOR_REQUEST
// OutBuffer: data, variable size!



//
// Device related Driver Parameters
//

#define IOCTL_USBIO_GET_DEVICE_PARAMETERS       _USBIO_IOCTL_CODE(15,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: USBIO_DEVICE_PARAMETERS                                                 

#define IOCTL_USBIO_SET_DEVICE_PARAMETERS       _USBIO_IOCTL_CODE(16,METHOD_BUFFERED)
// InBuffer:  USBIO_DEVICE_PARAMETERS
// OutBuffer: none



//
// Other 
//

#define IOCTL_USBIO_GET_CONFIGURATION_INFO      _USBIO_IOCTL_CODE(20,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: USBIO_CONFIGURATION_INFO

#define IOCTL_USBIO_RESET_DEVICE                _USBIO_IOCTL_CODE(21,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: none

#define IOCTL_USBIO_GET_CURRENT_FRAME_NUMBER    _USBIO_IOCTL_CODE(22,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: USBIO_FRAME_NUMBER


#define IOCTL_USBIO_SET_DEVICE_POWER_STATE      _USBIO_IOCTL_CODE(23,METHOD_BUFFERED)
// InBuffer:  USBIO_DEVICE_POWER
// OutBuffer: none

#define IOCTL_USBIO_GET_DEVICE_POWER_STATE      _USBIO_IOCTL_CODE(24,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: USBIO_DEVICE_POWER


#define IOCTL_USBIO_GET_DRIVER_INFO             _USBIO_IOCTL_CODE(27,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: USBIO_DRIVER_INFO


#define IOCTL_USBIO_CYCLE_PORT                  _USBIO_IOCTL_CODE(28,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: none


//
// Pipe related Requests
//

#define IOCTL_USBIO_BIND_PIPE                   _USBIO_IOCTL_CODE(30,METHOD_BUFFERED)
// InBuffer:  USBIO_BIND_PIPE
// OutBuffer: none

#define IOCTL_USBIO_UNBIND_PIPE                 _USBIO_IOCTL_CODE(31,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: none



#define IOCTL_USBIO_RESET_PIPE                  _USBIO_IOCTL_CODE(32,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: none

#define IOCTL_USBIO_ABORT_PIPE                  _USBIO_IOCTL_CODE(33,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: none



#define IOCTL_USBIO_GET_PIPE_PARAMETERS         _USBIO_IOCTL_CODE(35,METHOD_BUFFERED)
// InBuffer:  none
// OutBuffer: USBIO_PIPE_PARAMETERS

#define IOCTL_USBIO_SET_PIPE_PARAMETERS         _USBIO_IOCTL_CODE(36,METHOD_BUFFERED)
// InBuffer:  USBIO_PIPE_PARAMETERS
// OutBuffer: none



#define IOCTL_USBIO_PIPE_CONTROL_TRANSFER_IN    _USBIO_IOCTL_CODE(40,METHOD_OUT_DIRECT)
// InBuffer:  USBIO_PIPE_CONTROL_TRANSFER
// OutBuffer: data, variable size!

#define IOCTL_USBIO_PIPE_CONTROL_TRANSFER_OUT   _USBIO_IOCTL_CODE(41,METHOD_IN_DIRECT)
// InBuffer:  USBIO_PIPE_CONTROL_TRANSFER
// OutBuffer: data, variable size!



//
// options (device related)
//
#define USBIO_RESET_DEVICE_ON_CLOSE   0x00000001
#define USBIO_UNCONFIGURE_ON_CLOSE    0x00000002
#define USBIO_ENABLE_REMOTE_WAKEUP    0x00000004

//
// flags (pipe related)
//
#define USBIO_SHORT_TRANSFER_OK       0x00010000
#define USBIO_START_TRANSFER_ASAP     0x00020000

//
// max array size
//
#define USBIO_MAX_INTERFACES      32
#define USBIO_MAX_PIPES           32



// pack the following structures                                                     
#include <PSHPACK1.H>


typedef enum _USBIO_PIPE_TYPE {
  PipeTypeControl = 0,
  PipeTypeIsochronous,
  PipeTypeBulk,
  PipeTypeInterrupt,
  // The following dummy value is included to force the compiler to make the 
  // size of the enum 4 bytes (32 bits). Some compilers, Borland C++ for 
  // instance, use the smallest possible integer for enums.
  PipeType_Force32bitEnum = 2000000000
} USBIO_PIPE_TYPE;


typedef enum _USBIO_REQUEST_RECIPIENT {
  RecipientDevice = 0,
  RecipientInterface,
  RecipientEndpoint,
  RecipientOther,
  // The following dummy value is included to force the compiler to make the 
  // size of the enum 4 bytes (32 bits). Some compilers, Borland C++ for 
  // instance, use the smallest possible integer for enums.
  Recipient_Force32bitEnum = 2000000000
} USBIO_REQUEST_RECIPIENT;


typedef enum _USBIO_REQUEST_TYPE {
  RequestTypeClass = 1,
  RequestTypeVendor,
  // The following dummy value is included to force the compiler to make the 
  // size of the enum 4 bytes (32 bits). Some compilers, Borland C++ for 
  // instance, use the smallest possible integer for enums.
  RequestType_Force32bitEnum = 2000000000
} USBIO_REQUEST_TYPE;


typedef enum _USBIO_DEVICE_POWER_STATE {
  DevicePowerStateD0 = 0,
  DevicePowerStateD1,
  DevicePowerStateD2,
  DevicePowerStateD3,
  // The following dummy value is included to force the compiler to make the 
  // size of the enum 4 bytes (32 bits). Some compilers, Borland C++ for 
  // instance, use the smallest possible integer for enums.
  DevicePowerState_Force32bitEnum = 2000000000
} USBIO_DEVICE_POWER_STATE;



//
// GET_DRIVER_INFO
//
typedef struct _USBIO_DRIVER_INFO {
  USHORT APIVersion;    // [MAJOR][MINOR] (BCD format)
  USHORT DriverVersion; // [MAJOR][MINOR]
  ULONG  DriverBuildNumber;
  ULONG  Flags;         // USBIO_INFOFLAG_XXX
} USBIO_DRIVER_INFO;

#define USBIO_INFOFLAG_CHECKED_BUILD    0x00000010
#define USBIO_INFOFLAG_DEMO_VERSION     0x00000020
#define USBIO_INFOFLAG_LIGHT_VERSION    0x00000100



//
// GET_DESCRIPTOR
// SET_DESCRIPTOR
//
typedef struct _USBIO_DESCRIPTOR_REQUEST {
  USBIO_REQUEST_RECIPIENT Recipient;
  UCHAR   DescriptorType;
  UCHAR   DescriptorIndex;
  USHORT  LanguageId;
} USBIO_DESCRIPTOR_REQUEST;

//
// SET_FEATURE
// CLEAR_FEATURE
//
typedef struct _USBIO_FEATURE_REQUEST {
  USBIO_REQUEST_RECIPIENT Recipient;
  USHORT  FeatureSelector;
  USHORT  Index;
} USBIO_FEATURE_REQUEST;


//
// GET_STATUS
//
typedef struct _USBIO_STATUS_REQUEST {
  USBIO_REQUEST_RECIPIENT Recipient;
  USHORT  Index;
} USBIO_STATUS_REQUEST;

typedef struct _USBIO_STATUS_REQUEST_DATA {
  USHORT  Status;
} USBIO_STATUS_REQUEST_DATA;


//
// GET_CONFIGURATION
//
typedef struct _USBIO_GET_CONFIGURATION_DATA {
  UCHAR ConfigurationValue;
} USBIO_GET_CONFIGURATION_DATA;


//
// GET_INTERFACE
//
typedef struct _USBIO_GET_INTERFACE {
  USHORT Interface;
} USBIO_GET_INTERFACE;

typedef struct _USBIO_GET_INTERFACE_DATA {
  UCHAR AlternateSetting;
} USBIO_GET_INTERFACE_DATA;



//
// SET_CONFIGURATION
// SET_INTERFACE
//
typedef struct _USBIO_INTERFACE_SETTING {
  USHORT  InterfaceIndex;
  USHORT  AlternateSettingIndex;
  ULONG   MaximumTransferSize;
} USBIO_INTERFACE_SETTING;

typedef struct _USBIO_SET_CONFIGURATION {
  USHORT ConfigurationIndex;
  USHORT NbOfInterfaces;
  USBIO_INTERFACE_SETTING InterfaceList[USBIO_MAX_INTERFACES];
} USBIO_SET_CONFIGURATION;



//
// CLASS_OR_VENDOR_IN_REQUEST
// CLASS_OR_VENDOR_OUT_REQUEST
//
typedef struct _USBIO_CLASS_OR_VENDOR_REQUEST {
  ULONG   Flags;
  USBIO_REQUEST_TYPE Type;
  USBIO_REQUEST_RECIPIENT Recipient;
  UCHAR   RequestTypeReservedBits;
  UCHAR   Request;
  USHORT  Value;
  USHORT  Index;
} USBIO_CLASS_OR_VENDOR_REQUEST;



//
// GET_DEVICE_PARAMETERS
// SET_DEVICE_PARAMETERS
//
typedef struct _USBIO_DEVICE_PARAMETERS {
  ULONG Options;
  ULONG RequestTimeout;
} USBIO_DEVICE_PARAMETERS;



//
// GET_CONFIGURATION_INFO
//
typedef struct _USBIO_INTERFACE_CONFIGURATION_INFO {
  UCHAR InterfaceNumber;
  UCHAR AlternateSetting;
  UCHAR Class;
  UCHAR SubClass;
  UCHAR Protocol;
  UCHAR NumberOfPipes;
  UCHAR reserved1;  // reserved fields, set to zero
  UCHAR reserved2;
} USBIO_INTERFACE_CONFIGURATION_INFO;

typedef struct _USBIO_PIPE_CONFIGURATION_INFO {
  USBIO_PIPE_TYPE PipeType;  // type
  ULONG  MaximumTransferSize;// maximum Read/Write buffer size 
  USHORT MaximumPacketSize;  // of the endpoint  
  UCHAR  EndpointAddress;    // including direction bit
  UCHAR  Interval;           // in ms for interrupt pipe
  UCHAR  InterfaceNumber;    // interface that the EP belongs to
  UCHAR  reserved1;  // reserved fields, set to zero
  UCHAR  reserved2;
  UCHAR  reserved3;
} USBIO_PIPE_CONFIGURATION_INFO;

typedef struct _USBIO_CONFIGURATION_INFO {
  ULONG NbOfInterfaces;
  ULONG NbOfPipes;
  USBIO_INTERFACE_CONFIGURATION_INFO InterfaceInfo[USBIO_MAX_INTERFACES];
  USBIO_PIPE_CONFIGURATION_INFO      PipeInfo[USBIO_MAX_PIPES];
} USBIO_CONFIGURATION_INFO;



//
// GET_CURRENT_FRAME_NUMBER
//
typedef struct _USBIO_FRAME_NUMBER {
  ULONG FrameNumber;
} USBIO_FRAME_NUMBER;


//
// SET_DEVICE_POWER_STATE
// GET_DEVICE_POWER_STATE
//
typedef struct _USBIO_DEVICE_POWER {
  USBIO_DEVICE_POWER_STATE DevicePowerState;
} USBIO_DEVICE_POWER;



//
// BIND_PIPE
//
typedef struct _USBIO_BIND_PIPE {
  UCHAR EndpointAddress; // including direction bit
} USBIO_BIND_PIPE;


//
// GET_PIPE_PARAMETERS
// SET_PIPE_PARAMETERS
//
typedef struct _USBIO_PIPE_PARAMETERS {
  ULONG Flags;
} USBIO_PIPE_PARAMETERS;


//
// PIPE_CONTROL_TRANSFER_IN
// PIPE_CONTROL_TRANSFER_OUT
//
typedef struct _USBIO_PIPE_CONTROL_TRANSFER {
  ULONG Flags;
  UCHAR SetupPacket[8];
} USBIO_PIPE_CONTROL_TRANSFER;



//
// Isochronous Transfers
//
// The data buffer passed to ReadFile/WriteFile must contain a
// predefined header that describes the size and location of the 
// packets to be transferred. The USBIO_ISO_TRANSFER_HEADER consists
// of a fixed size part USBIO_ISO_TRANSFER and a variable size array
// of USBIO_ISO_PACKET descriptors.
//

typedef struct _USBIO_ISO_TRANSFER {
  ULONG NumberOfPackets;
  ULONG Flags;
  ULONG StartFrame;
  ULONG ErrorCount;
} USBIO_ISO_TRANSFER;

typedef struct _USBIO_ISO_PACKET {
  ULONG Offset;
  ULONG Length;
  ULONG Status;   
} USBIO_ISO_PACKET;

typedef struct _USBIO_ISO_TRANSFER_HEADER {
  USBIO_ISO_TRANSFER  IsoTransfer;
  USBIO_ISO_PACKET    IsoPacket[1]; // variable size array
} USBIO_ISO_TRANSFER_HEADER;



// restore packing
#include <POPPACK.H>


/********************** END OF INTERFACE ********************************/

//
// Define the device type value. Note that values used by Microsoft
// are in the range 0-32767, and 32768-65535 are reserved for use
// by customers.
//
#define FILE_DEVICE_USBIO       0x8094

//
// Macros to generate IOCTL codes.
// Note that function codes 0-2047 are reserved for Microsoft, and
// 2048-4095 are reserved for customers.
//
#define _USBIO_IOCTL_BASE       0x800

#define _USBIO_IOCTL_CODE(FnCode,Method)    \
   ( (ULONG)CTL_CODE(                       \
      (ULONG)FILE_DEVICE_USBIO,             \
      (ULONG)(_USBIO_IOCTL_BASE+(FnCode)),  \
      (ULONG)(Method),                      \
      (ULONG)FILE_ANY_ACCESS                \
      ) )


#endif  // _USBIO_I_H_

/*************************** EOF **************************************/
