/************************************************************************
 *
 *  Module:       UsbIoThread.h
 *  Long name:    CUsbIoThread class
 *  Description:  Input/Output Thread, base class for Reader and Writer
 *
 *  Runtime Env.: Win32, Part of UsbioLib
 *  Author(s):    Guenter Hildebrandt, Udo Eberhardt
 *  Company:      Thesycon GmbH, Ilmenau
 ************************************************************************/

#ifndef _UsbIoThread_h_
#define _UsbIoThread_h_

#include "UsbIoPipe.h"



//
// CUsbIoThread
//
// This class provides a basic implementation of a worker thread
// that is used to perform continuous I/O operations.
// It is used as a base class for Reader and Writer worker thread implementations.
//
class CUsbIoThread : public CUsbIoPipe
{
public:
  // standard constructor
  CUsbIoThread();
  // destructor
  virtual ~CUsbIoThread();

  // allocate the internal buffer pool
  // returns TRUE for success, FALSE if a memory allocation fails
  BOOL AllocateBuffers(DWORD SizeOfBuffer, DWORD NumberOfBuffers);
  
  // free all elements of the internal buffer pool
  void FreeBuffers();
  
  // start the worker thread
  BOOL StartThread(DWORD MaxIoErrorCount = 3);
  
  // terminate the worker thread
  // The function sets TerminateFlag to TRUE, calls the virtual member function
  // TerminateThread() that should cause the thread to terminate itself, and then
  // blocks until the worker thread has been terminated.
  // IMPORTANT: This function must be called before the CUsbIoThread object is
  // destroyed, that means before ~CUsbIoThread() is called !
  BOOL ShutdownThread();


  // This notification handler is called by the worker thread before
  // the thread terminates itself.
  // NOTE: It is called in the context of the worker thread.
  virtual void OnThreadExit();


  // This routine is called by the thread if a buffer must be
  // filled with data before it will be send to the pipe.
  // It should be overloaded by a derived class and implement 
  // the buffer processing.
  // The standard implementation zeroes the buffer and sets the 
  // NumberOfBytesToTransfer to BufferSize, but does nothing else.
  virtual void ProcessBuffer(CUsbIoBuf* Buf);

  // This routine is called by the thread if a buffer
  // was successfully read from the pipe. It should be overloaded
  // by a derived class and implement the processing of the data.
  // The standard implementation does nothing.
  virtual void ProcessData(CUsbIoBuf* Buf);

  // This routine is called if an error occurs while writing
  // or reading a buffer to/from the USB device.
  // It should be overloaded by a derived class and implement 
  // the appropriate error handling.
  // The standard implementation does nothing.
  virtual void BufErrorHandler(CUsbIoBuf* Buf);


// implementation:
protected:

  // the routine that is executed by the thread
  // must be implemented by derived classes
  virtual void ThreadRoutine() = 0;
  
  // this routine is called to terminate the thread
  // must be implemented by derived classes
  virtual void TerminateThread() = 0;

  // system thread routine
  static unsigned int __stdcall beginthread_routine(void *StartContext);

  // internal buffer pool
  CUsbIoBufPool BufPool;
  // thread
  HANDLE ThreadHandle;
  unsigned int ThreadID;

  // variables used by the thread routine
  volatile BOOL TerminateFlag;
  DWORD MaxErrorCount;
  // list of pending buffers
  CUsbIoBuf *FirstPending;
  CUsbIoBuf *LastPending;

}; // class CUsbIoThread



#endif // _UsbIoThread_h_
 
/*************************** EOF **************************************/
