/************************************************************************
 *
 *  Module:       UsbIoPipe.h
 *  Long name:    CUsbIoPipe class
 *  Description:  USB In/Out Pipe class definition
 *
 *  Runtime Env.: Win32, Part of UsbioLib
 *  Author(s):    Guenter Hildebrandt, Udo Eberhardt
 *  Company:      Thesycon GmbH, Ilmenau
 ************************************************************************/

#ifndef _UsbIoPipe_h_
#define _UsbIoPipe_h_

#include "UsbIo.h"
#include "UsbIoBuf.h"



//
// CUsbIoPipe
//
// This class implements the interface to a USB Pipe that is exported
// by the usbio.sys device driver.
// It contains all pipe-related functions that can be executed only
// on a file handle that is bound to an endpoint.
//
class CUsbIoPipe : public CUsbIo
{
public:
  // standard constructor
  CUsbIoPipe();
  // destructor
  virtual ~CUsbIoPipe();

  // bind the pipe to an endpoint
  // The pipe can be bound to exactly one endpoint.
  // The endpoint address must contain the direction bit.
  // The endpoint must be in the actual configuration.
  // Bind calls CUsbIo::Open() with the parameters specified if the device is not already open.
  // For a description of the meaning of the parameters DeviceNumber, DeviceList, 
  // and InterfaceGuid see CUsbIo::Open() in UsbIo.h.
  DWORD Bind(int DeviceNumber, UCHAR EndpointAddress, HDEVINFO DeviceList = NULL, const GUID *InterfaceGuid = NULL);

  // delete the binding to an endpoint
  DWORD Unbind();


  // reset a pipe after an error condition
  // Use this function to clean the error state 'pipe halted'.
  DWORD ResetPipe();

  // cancel all pending read/write requests on this pipe
  DWORD AbortPipe();


  // get/set pipe-related parameters
  // Note: DataRate is valid for isochronous pipes only.
  // It is ignored for all other pipe types.
  // The unit of DataRate is bytes per second.
  // The maximum DataRate is (PipeMaxPacketSize*1000).
  DWORD GetPipeParameters(USBIO_PIPE_PARAMETERS* PipeParameters);
  DWORD SetPipeParameters(const USBIO_PIPE_PARAMETERS* PipeParameters);


  // issue a control transfer (setup token) on this pipe
  // Note: It is *not* possible to generate a setup token for
  // the default pipe (EP0).
  // Use pipe control for user defined setup tokens only.
  //
  // ByteCount is an IN and OUT parameter
  // it contains the size of the buffer (IN)
  // and the number of bytes transferred (OUT) if the function call was successful
  //
  // direction: Device to Host (in)
  DWORD PipeControlTransferIn(
            void* Buffer,
            DWORD& ByteCount,
            const USBIO_PIPE_CONTROL_TRANSFER* ControlTransfer);
  // direction: Host to Device (out)
  DWORD PipeControlTransferOut(
            const void* Buffer,
            DWORD& ByteCount,
            const USBIO_PIPE_CONTROL_TRANSFER* ControlTransfer);



  // issue a read or write request on this pipe
  // Read and Write can be used for all pipe types.
  // The call is always handled asynchronously.
  // The function returns TRUE if the request was successfully
  // submitted to the driver. The return value is FALSE if 
  // an error is occured, the error code is in the Status member
  // of CUsbIoBuf.
  BOOL Read(CUsbIoBuf* Buf);
  BOOL Write(CUsbIoBuf* Buf);


  // wait for completion of a read or write operation
  // After submitting a buffer using Read or Write, this function
  // may be called with this buffer to wait for completion
  // of the data transfer.
  // WaitForCompletion can be called regardless of the return status
  // of the Read or Write function. It returns always the final
  // completion status of the operation.
  // Optionally a timeout interval for the wait may be specified.
  // A value of INFINITE means: no timeout (wait until the operation completes).
  // Note: After a status of USBIO_ERR_TIMEOUT was returned, the operation
  // is still pending and the WaitForCompletion can be called again.
  DWORD WaitForCompletion(CUsbIoBuf* Buf, DWORD Timeout=INFINITE);


  // write or read to/from this pipe using synchronous operation
  // The functions transfer data from or to the specified buffer.
  // Optionally a timeout interval for the completion of the transfer may be specified.
  // A value of INFINITE means: no timeout (wait until the operation completes).
  // The ReadSync and WriteSync functions do not return to the caller until 
  // the data transfer is completed, with success or error, or a timeout
  // error occurs. In the case of a timeout the read or write operation
  // is aborted (cancelled) and the status USBIO_ERR_TIMEOUT is returned.
  //
  // ByteCount is an IN and OUT parameter
  // it contains the size of the buffer (IN)
  // and the number of bytes transferred (OUT) if the function call was successful
  //
  // NOTE: There is some overhead when using the sync. functions.
  // Every time the functions are called an event object will be created
  // internally (with CreateEvent) and destroyed again.
  //
  DWORD ReadSync(
          void *Buffer,
          DWORD &ByteCount,
          DWORD Timeout=INFINITE );

  DWORD WriteSync(
          void *Buffer,
          DWORD &ByteCount,
          DWORD Timeout=INFINITE );


}; // class CUsbIoPipe


#endif // _UsbIoPipe_h_
 
/*************************** EOF **************************************/
