/************************************************************************
 *
 *  Module:       UsbIoBuf.h
 *  Long name:    CUsbIoBuf class
 *  Description:  definition of an USB I/O buffer and buffer pool
 *
 *  Runtime Env.: Win32, Part of UsbioLib
 *  Author(s):    Guenter Hildebrandt, Udo Eberhardt
 *  Company:      Thesycon GmbH, Ilmenau
 ************************************************************************/

#ifndef _UsbIoBuf_h_
#define _UsbIoBuf_h_



//
// CUsbIoBuf
//
// This class is used as an buffer descriptor for read and write
// buffers.
//
class CUsbIoBuf
{
  friend class CUsbIoBufPool;
  friend class CUsbIoPipe;
public:
  // standard constructor
  CUsbIoBuf();
  // construct and use existing buffer
  CUsbIoBuf(void *buffer, DWORD buffersize);
  // construct and allocate a buffer internally
  CUsbIoBuf(DWORD buffersize);
  // destructor
  ~CUsbIoBuf();

  // get pointer to first byte of buffer
  void* Buffer()  { return BufferMem; }

  // get size of buffer in bytes
  DWORD Size()    { return BufferSize; }

  // number of bytes to transfer from/to buffer
  // must be set *before* Read or Write is called
  DWORD NumberOfBytesToTransfer;

  // number of bytes transferred from/to buffer
  // will be set *after* the read or write operation was completed
  DWORD BytesTransferred;

  // final status of the read or write operation
  // will be set *after* the read or write operation was completed
  DWORD Status;

  // link pointer, may be used to build a chain of buffer objects
  // used by CUsbIoBufPool and CUsbIoReader/CUsbIoWriter
  CUsbIoBuf *Next;                

  // set this flag to true in the ProcessBuffer routine, if the 
  // read or write operation is finished, otherwise to false
  BOOL OperationFinished;

// implementation:
protected:

  void Init();

  void *BufferMem;    // pointer to the buffer
  DWORD BufferSize;   // size of the allocated buffer in bytes
  OVERLAPPED Overlapped;  // overlapped struct used for asynch. IO
  BOOL BufferMemAllocated;  // TRUE: buffer was allocated by the CUsbIoBuf

};



//
// CUsbIoBufPool
//
// This class implements a pool of CUsbIoBuf objects.
// It is used by CUsbIoReader and CUsbIoWriter as a buffer pool.
//
class CUsbIoBufPool
{
public:
  // standard constructor
  CUsbIoBufPool();
  // destructor
  ~CUsbIoBufPool();

  // allocate all elements of the buffer pool
  // returns TRUE for success, FALSE if a memory allocation fails
  BOOL Allocate(DWORD SizeOfBuffer, DWORD NumberOfBuffers);

  // free all elements of the buffer pool
  void Free();

  // returns the current number of buffers in pool
  int CurrentCount()    { return Count; }

  // get a buffer from pool
  // returns NULL if the pool is empty
  CUsbIoBuf* Get();
  
  // put a buffer back to pool
  void Put(CUsbIoBuf* Buf);


// implementation:
protected:
  CUsbIoBuf *Head;  // head pointer of buffer chain
  int Count;        // current number of buffers in pool
  CRITICAL_SECTION CritSect;
  CUsbIoBuf *BufArray;
  char *BufferMemory;

};


#endif // _UsbIoBuf_h_
 
/*************************** EOF **************************************/
