/************************************************************************
 *
 *  Module:       UsbIo.h
 *  Long name:    CUsbIo class
 *  Description:  CUsbIo base device class definition
 *
 *  Runtime Env.: Win32, Part of UsbioLib
 *  Author(s):    Guenter Hildebrandt, Udo Eberhardt
 *  Company:      Thesycon GmbH, Ilmenau
 ************************************************************************/

#ifndef _UsbIo_h_
#define _UsbIo_h_

// get Win32 IOCTL support
#include <winioctl.h>
// get setup API functions (only available in Win98 and Win2K)
// requires to link with setupapi.lib
#include <setupapi.h>

// get driver interface definitions
#include "usbspec.h"
#include "usbio_i.h"


//
// Device names created by the USBIO driver
//
// NOTES:
//
// !!!!!!!!!!!!
// This device names should not be used anymore. Use it only if compatibility to
// earlier versions of USBIO is required.
// Use device interfaces identified by GUIDs instead!
// If the old-style device names (USBIO_DeviceX) are required to ensure
// backward-compatibility to earlier versions of USBIO they must be enabled
// explicitly in USBIO.INF. See comments in USBIO.INF.
// !!!!!!!!!!!!
//
//
// old-style device naming:
//
// The string defined here MUST match to the string defined by the
// USBIO driver configuration parameter 'USBIO_DeviceBaseName' in the registry.
//
// One digit must be appended for device enumeration.
// e.g. \\.\USBIO_Device0, \\.\USBIO_Device1
//
#define USBIO_DEVICE_NAME      "USBIO_Device"



//
// CUsbIo
//
// This class implements the interface to the usbio.sys device driver.
// It contains only general device-related functions that can be executed
// without a pipe context.
// Pipe specific functions are implemented by the CUsbIoPipe class.
//
class CUsbIo
{
public:
  // standard constructor
  CUsbIo();
  // destructor, should be virtual
  virtual ~CUsbIo();

  // Create a windows-internal device list with all matching interfaces.
  // The device interface is identified by InterfaceGuid. A handle for the list
  // is returned in case of success, or NULL is returned in case of error.
  // The device list can be iterated with Open(), see below.
  // The device list must be freed with DestroyDeviceList().
  static HDEVINFO CreateDeviceList(const GUID* InterfaceGuid);
  // Use this function to destroy the device list, generated with CreateDeviceList
  static void DestroyDeviceList(HDEVINFO DeviceList);

  
  // Open an USB device
  // 
  // There are two options:
  //
  // (a) DeviceList==NULL
  // The parameter InterfaceGuid will be ignored. DeviceNumber will be used to build
  // an old-style device name that starts with the string defined by USBIO_DEVICE_NAME
  // (see comments on USBIO_DEVICE_NAME above).
  // Note: This mode should be used only if compatibility to earlier versions of USBIO
  // is required! It will work only if the creation of static device names is enabled in
  // the USBIO.INF file. It is not recommended to use this mode.
  // 
  // (b) DeviceList!=NULL
  // The device list provided in DeviceList must be build using CreateDeviceList() (see above).
  // The GUID that identifies the device interface must be provided in InterfaceGuid.
  // DeviceNumber is used to iterate through the device list. It should start with zero and
  // should be incremented after each call to Open(). If no more instances of the interface 
  // are available USBIO_ERR_NO_SUCH_DEVICE_INSTANCE is returned.
  // Note: This is the recommended way for device enumeration.
  //
  DWORD Open(int DeviceNumber, HDEVINFO DeviceList =NULL, const GUID* InterfaceGuid =NULL);

  // close the device
  // can be called if the device is not open, does nothing in this case
  void Close();


  // Returns a pointer to the path name associated with this device instance.
  // The pointer is temporarily valid and should not be stored for later use. 
  // It becomes invalid if the device is closed.
  // If no device is opened, the return value is NULL.
  // The return value is always NULL if the device is opened using 
  // case (a) described above.
  const char* GetDevicePathName();
  
  
  // get driver info
  // API version, driver version, build number, ...
  DWORD GetDriverInfo(USBIO_DRIVER_INFO *DriverInfo);
  
  
  // get descriptor
  // ByteCount is an IN and OUT parameter
  // it contains the size of the buffer (IN)
  // and the number of bytes returned (OUT) if the function call was successful
  DWORD GetDescriptor(
            void* Buffer,
            DWORD& ByteCount,
            USBIO_REQUEST_RECIPIENT Recipient,
            UCHAR DescriptorType,
            UCHAR DescriptorIndex=0,
            USHORT LanguageId=0 );

  // set descriptor
  // ByteCount is an IN and OUT parameter
  // it contains the size of the buffer (IN)
  // and the number of bytes transferred (OUT) if the function call was successful
  DWORD SetDescriptor(
            const void* Buffer,
            DWORD& ByteCount,
            USBIO_REQUEST_RECIPIENT Recipient,
            UCHAR DescriptorType,
            UCHAR DescriptorIndex=0,
            USHORT LanguageId=0 );


  // set feature
  DWORD SetFeature(
            USBIO_REQUEST_RECIPIENT Recipient,
            USHORT FeatureSelector,
            USHORT Index=0 );

  // clear feature
  DWORD ClearFeature(
            USBIO_REQUEST_RECIPIENT Recipient,
            USHORT FeatureSelector,
            USHORT Index=0 );


  // get status
  // StatusValue returns the status data if the call was successful
  DWORD GetStatus(
            USHORT& StatusValue,
            USBIO_REQUEST_RECIPIENT Recipient,
            USHORT Index=0 );


  // get current configuration
  // ConfigurationValue returns the current configuration if the call was successful
  DWORD GetConfiguration(
            UCHAR& ConfigurationValue );
  
  // get current alternate setting of interface
  // AlternateSetting returns the current setting if the call was successful
  DWORD GetInterface(
            UCHAR& AlternateSetting,
            USHORT Interface=0 );
  // store a configuration descriptor and use it for the next SetConfiguration()
  DWORD StoreConfigurationDescriptor(const USB_CONFIGURATION_DESCRIPTOR *Desc);
  // set configuration
  DWORD SetConfiguration(const USBIO_SET_CONFIGURATION* Conf);
  // delete current configuration
  DWORD UnconfigureDevice();
  // set interface
  DWORD SetInterface(const USBIO_INTERFACE_SETTING* Setting);


  // class or vendor specific request, direction: Device to Host (in)
  // ByteCount is an IN and OUT parameter
  // it contains the size of the buffer (IN)
  // and the number of bytes returned (OUT) if the function call was successful
  DWORD ClassOrVendorInRequest(
            void* Buffer,
            DWORD& ByteCount,
            const USBIO_CLASS_OR_VENDOR_REQUEST *Request );

  // class or vendor specific request, direction: Host to Device (out)
  // ByteCount is an IN and OUT parameter
  // it contains the size of the buffer (IN)
  // and the number of bytes transferred (OUT) if the function call was successful
  DWORD ClassOrVendorOutRequest(
            const void* Buffer,
            DWORD& ByteCount,
            const USBIO_CLASS_OR_VENDOR_REQUEST *Request );


  // get device-related driver parameters
  DWORD GetDeviceParameters(USBIO_DEVICE_PARAMETERS *DevParam);

  // set device-related driver parameters
  DWORD SetDeviceParameters(const USBIO_DEVICE_PARAMETERS *DevParam);


  // get device configuration information
  DWORD GetConfigurationInfo(USBIO_CONFIGURATION_INFO *Info);
  
  
  // reset device (Hub Port Reset)
  // abort and unbind all pipes
  // device will be set into unconfigured state
  DWORD ResetDevice();

  // simulates a device disconnect/connect cycle
  // this will unload/load the USBIO driver
  DWORD CyclePort();


  // get current USB frame number from host controller
  // FrameNumber contains the current frame number if the call was successful
  DWORD GetCurrentFrameNumber(DWORD &FrameNumber);

  
  // get current device power state
  DWORD GetDevicePowerState(USBIO_DEVICE_POWER_STATE &DevicePowerState);
  
  // set device to power state
  DWORD SetDevicePowerState(USBIO_DEVICE_POWER_STATE DevicePowerState);


  //
  // support for standard descriptors
  // Implemented as wrappers for GetDescriptor().
  // For a description, see GetDescriptor()
  //

  DWORD GetDeviceDescriptor(
            USB_DEVICE_DESCRIPTOR* Desc );

  DWORD GetConfigurationDescriptor(
            USB_CONFIGURATION_DESCRIPTOR* Desc,
            DWORD& ByteCount,
            UCHAR Index=0 );

  DWORD GetStringDescriptor(
            USB_STRING_DESCRIPTOR* Desc,
            DWORD& ByteCount,
            UCHAR Index=0,
            USHORT LanguageId=0 );



  // translate an error code to a readable string
  // The function prints to StringBuffer and returns a pointer to StringBuffer.
  static char* ErrorText(char* StringBuffer, DWORD StringBufferSize, DWORD ErrorCode);


  // call a driver IOCTL function and wait for completion
  DWORD IoctlSync(
            DWORD IoctlCode,
            const void *InBuffer,
            DWORD InBufferSize,
            void *OutBuffer,
            DWORD OutBufferSize,
            DWORD *BytesReturned
            );

  // cancel all outstanding requests that were
  // issued by the calling thread on the file handle
  // Note: Requests issued on the handle by other threads are NOT cancelled.
  BOOL CancelIo();


  // returns TRUE if the class instance is attached to a device
  BOOL IsOpen() { return (FileHandle!=NULL); }

  // returns TRUE if a checked build (debug version) of the USBIO driver was detected
  BOOL IsCheckedBuild() { return CheckedBuildDetected; }

  // returns TRUE if a demo version of the USBIO driver was detected
  BOOL IsDemoVersion()  { return DemoVersionDetected; }


// implementation
protected:      

  HANDLE FileHandle;
  OVERLAPPED Overlapped;
  CRITICAL_SECTION CritSect;

  BOOL CheckedBuildDetected;
  BOOL DemoVersionDetected;

private:
  SP_DEVICE_INTERFACE_DETAIL_DATA* mDevDetail;

}; // class CUsbIo


#endif // _UsbIo_h_
 
/*************************** EOF **************************************/
