/************************************************************************
 *
 ************************************************************************/

// for shorter and faster windows.h
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN 1
#endif
// unicode is not supported by USBIOLIB
#ifdef UNICODE
#undef UNICODE
#endif

#include <windows.h>
#include <stdio.h>
#include "usbio.h"

// standard constructor
CUsbIo::CUsbIo()
{
  FileHandle = NULL;
  ZeroMemory(&Overlapped,sizeof(Overlapped));
  InitializeCriticalSection(&CritSect);
  CheckedBuildDetected = FALSE;
  DemoVersionDetected = FALSE;
  mDevDetail=NULL;
}


// destructor
CUsbIo::~CUsbIo()
{
  // close file handle
  Close();
  // free resources
  DeleteCriticalSection(&CritSect);

}


HDEVINFO
CUsbIo::CreateDeviceList(const GUID *InterfaceGuid)
{
  HDEVINFO h;

  h = SetupDiGetClassDevs(
        (GUID*)InterfaceGuid,                 // LPGUID ClassGuid, 
        NULL,                                 // PCTSTR Enumerator, 
        NULL,                                 // HWND hwndParent, 
        DIGCF_DEVICEINTERFACE | DIGCF_PRESENT // DWORD Flags
        );
  return ( (h==INVALID_HANDLE_VALUE) ? NULL : h );
}


void 
CUsbIo::DestroyDeviceList(HDEVINFO DeviceList)
{
  if (DeviceList!=NULL) {
    SetupDiDestroyDeviceInfoList(DeviceList);
  }
}



DWORD CUsbIo::Open(int DeviceNumber, HDEVINFO DeviceList /*=NULL*/, const GUID *InterfaceGuid /*=NULL*/)
{
  DWORD Status;
  HANDLE h;
  char NameBuffer[80];
  char *Name;
  SP_DEVICE_INTERFACE_DATA  DevData;
  DWORD ReqLen;
  BOOL succ;

  if ( FileHandle != NULL ) {
    // already open
    return USBIO_ERR_SUCCESS;
  } 
  
  
  if (DeviceList == NULL) {
    // use the old way with a known name
    // build device name
    sprintf(NameBuffer,"\\\\.\\" USBIO_DEVICE_NAME "%d",DeviceNumber);
    Name = NameBuffer;
  } else {
    // use device interface identified by InterfaceGuid
    // a GUID must be provided in this case
    if ( InterfaceGuid==NULL ) {
      return USBIO_ERR_INVALID_FUNCTION_PARAM;
    }
    // delete old detail data if any
    if (mDevDetail!=NULL) {
      delete [] (char*)mDevDetail;
      mDevDetail = NULL;
    }
    // enumerate the interface
    // get the device information for the given device number
    ZeroMemory(&DevData,sizeof(DevData));
    DevData.cbSize = sizeof(SP_DEVICE_INTERFACE_DATA);
    succ = SetupDiEnumDeviceInterfaces(DeviceList, NULL, (GUID*)InterfaceGuid, DeviceNumber, &DevData );
    if (!succ) {
      Status=GetLastError();
      if ( Status==ERROR_NO_MORE_ITEMS ) Status = USBIO_ERR_NO_SUCH_DEVICE_INSTANCE;
      return Status;
    }
    // get length of the detailed information, allocate buffer
    SetupDiGetDeviceInterfaceDetail(DeviceList, &DevData, NULL, 0, &ReqLen, NULL);
    mDevDetail = (SP_DEVICE_INTERFACE_DETAIL_DATA*) new char[ReqLen];
    if (mDevDetail==NULL) {
      return USBIO_ERR_NO_MEMORY;
    }
    // now get the  detailed device information
    ZeroMemory(mDevDetail,ReqLen);
    mDevDetail->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);
    succ = SetupDiGetDeviceInterfaceDetail(DeviceList, &DevData, mDevDetail, ReqLen, &ReqLen, NULL);
    if (!succ) {
      return GetLastError();
    }
    // get name
    Name = mDevDetail->DevicePath;
  }

  // try to open the device driver
  h = ::CreateFile(
            Name,
            GENERIC_READ | GENERIC_WRITE,       // access mode
            FILE_SHARE_WRITE | FILE_SHARE_READ, // share mode
            NULL,                               // security desc.
            OPEN_EXISTING,                      // how to create
            FILE_FLAG_OVERLAPPED,               // file attributes
            NULL                                // template file
            );
  if ( h == INVALID_HANDLE_VALUE ) {
    Status = USBIO_ERR_DEVICE_NOT_FOUND;
  } else {
    // save handle
    FileHandle = h;
    // init the event with auto reset, not signaled
    Overlapped.hEvent = CreateEvent(NULL ,FALSE ,FALSE ,NULL); 
    if ( Overlapped.hEvent == NULL ) {
      Status = USBIO_ERR_NO_MEMORY;
      Close();
    } else {
      
      // now get version info
      USBIO_DRIVER_INFO info;
      Status = GetDriverInfo(&info);
      if ( Status != USBIO_ERR_SUCCESS ) {
        // failed
        Close();
      } else {

        CheckedBuildDetected = (info.Flags&USBIO_INFOFLAG_CHECKED_BUILD) ? TRUE : FALSE;
        DemoVersionDetected = (info.Flags&USBIO_INFOFLAG_DEMO_VERSION) ? TRUE : FALSE;

        // now check the API version
        // currently used version must match to the driver installed
        if ( info.APIVersion != USBIO_API_VERSION ) {
          // wrong version
          Status = USBIO_ERR_VERSION_MISMATCH;
          Close();
        } else {
          
          // success
          Status = USBIO_ERR_SUCCESS;
        }
      }
    }
  }
  return Status;
}



void CUsbIo::Close()
{
  if ( FileHandle != NULL ) {
    ::CloseHandle(FileHandle);
    FileHandle = NULL;
  }

  if ( Overlapped.hEvent != NULL ) {
    ::CloseHandle(Overlapped.hEvent);
    Overlapped.hEvent = NULL;
  }
  if (mDevDetail) {
    delete [] (char*)mDevDetail;
    mDevDetail = NULL;
  }
}


const char*
CUsbIo::GetDevicePathName() {
  if (mDevDetail) {
    return mDevDetail->DevicePath;
  } else {
    return NULL;
  }
}


DWORD CUsbIo::GetDriverInfo(USBIO_DRIVER_INFO *DriverInfo)
{
  DWORD Status;

  Status = IoctlSync(
              IOCTL_USBIO_GET_DRIVER_INFO,
              NULL,
              0,
              DriverInfo,
              sizeof(USBIO_DRIVER_INFO),
              NULL
              );

  return Status;
}



DWORD CUsbIo::GetDescriptor(
        void* Buffer,
        DWORD& ByteCount,
        USBIO_REQUEST_RECIPIENT Recipient,
        UCHAR DescriptorType,
        UCHAR DescriptorIndex/*=0*/,
        USHORT LanguageId/*=0*/ )
{
  DWORD Status;
  USBIO_DESCRIPTOR_REQUEST req;

  // zero the struct if any fields are added...
  ZeroMemory(&req,sizeof(req));
  req.Recipient = Recipient;
  req.DescriptorType = DescriptorType;
  req.DescriptorIndex = DescriptorIndex;
  req.LanguageId = LanguageId;
  
  Status = IoctlSync(
              IOCTL_USBIO_GET_DESCRIPTOR,
              &req,
              sizeof(req),
              Buffer,
              ByteCount,
              &ByteCount
              );

  return Status;
}



DWORD CUsbIo::SetDescriptor(
        const void* Buffer,
        DWORD& ByteCount,
        USBIO_REQUEST_RECIPIENT Recipient,
        UCHAR DescriptorType,
        UCHAR DescriptorIndex/*=0*/,
        USHORT LanguageId/*=0*/ )
{
  DWORD Status;
  USBIO_DESCRIPTOR_REQUEST req;

  // zero the struct if any fields are added...
  ZeroMemory(&req,sizeof(req));
  req.Recipient = Recipient;
  req.DescriptorType = DescriptorType;
  req.DescriptorIndex = DescriptorIndex;
  req.LanguageId = LanguageId;
  
  Status = IoctlSync(
              IOCTL_USBIO_SET_DESCRIPTOR,
              &req,
              sizeof(req),
              (void*)Buffer,
              ByteCount,
              &ByteCount
              );

  return Status;
}



DWORD CUsbIo::SetFeature(
        USBIO_REQUEST_RECIPIENT Recipient,
        USHORT FeatureSelector,
        USHORT Index/*=0*/ )
{
  DWORD Status;
  USBIO_FEATURE_REQUEST req;

  // zero the struct if any fields are added...
  ZeroMemory(&req,sizeof(req));
  req.Recipient = Recipient;
  req.FeatureSelector = FeatureSelector;
  req.Index = Index;
  
  Status = IoctlSync(
              IOCTL_USBIO_SET_FEATURE,
              &req,
              sizeof(req),
              NULL,
              0,
              NULL
              );

  return Status;
}



DWORD CUsbIo::ClearFeature(
        USBIO_REQUEST_RECIPIENT Recipient,
        USHORT FeatureSelector,
        USHORT Index/*=0*/ )
{
  DWORD Status;
  USBIO_FEATURE_REQUEST req;

  // zero the struct if any fields are added...
  ZeroMemory(&req,sizeof(req));
  req.Recipient = Recipient;
  req.FeatureSelector = FeatureSelector;
  req.Index = Index;
  
  Status = IoctlSync(
              IOCTL_USBIO_CLEAR_FEATURE,
              &req,
              sizeof(req),
              NULL,
              0,
              NULL
              );

  return Status;
}



DWORD CUsbIo::GetStatus(
        USHORT& StatusValue,
        USBIO_REQUEST_RECIPIENT Recipient,
        USHORT Index/*=0*/ )
{
  DWORD Status;
  USBIO_STATUS_REQUEST req;
  USBIO_STATUS_REQUEST_DATA data;

  // zero the structs if any fields are added...
  ZeroMemory(&req,sizeof(req));
  ZeroMemory(&data,sizeof(data));
  req.Recipient = Recipient;
  req.Index = Index;
  
  Status = IoctlSync(
              IOCTL_USBIO_GET_STATUS,
              &req,
              sizeof(req),
              &data,
              sizeof(data),
              NULL
              );

  StatusValue = data.Status;

  return Status;
}



DWORD CUsbIo::GetConfiguration(
        UCHAR& ConfigurationValue )
{
  DWORD Status;
  USBIO_GET_CONFIGURATION_DATA data;

  // zero the struct
  ZeroMemory(&data,sizeof(data));
  
  Status = IoctlSync(
              IOCTL_USBIO_GET_CONFIGURATION,
              NULL,
              0,
              &data,
              sizeof(data),
              NULL
              );

  ConfigurationValue = data.ConfigurationValue;

  return Status;
}



DWORD CUsbIo::GetInterface(
        UCHAR& AlternateSetting,
        USHORT Interface/*=0*/ )
{
  DWORD Status;
  USBIO_GET_INTERFACE req;
  USBIO_GET_INTERFACE_DATA data;

  // zero the structs if any fields are added...
  ZeroMemory(&req,sizeof(req));
  ZeroMemory(&data,sizeof(data));
  req.Interface = Interface;
  
  Status = IoctlSync(
              IOCTL_USBIO_GET_INTERFACE,
              &req,
              sizeof(req),
              &data,
              sizeof(data),
              NULL
              );

  AlternateSetting = data.AlternateSetting;

  return Status;
}


DWORD CUsbIo::StoreConfigurationDescriptor(const USB_CONFIGURATION_DESCRIPTOR *Desc)
{
  DWORD Status;

  Status = IoctlSync(
              IOCTL_USBIO_STORE_CONFIG_DESCRIPTOR,
              Desc,
              Desc->wTotalLength,
              NULL,
              0,
              NULL
              );

  return Status;
}


DWORD CUsbIo::SetConfiguration(const USBIO_SET_CONFIGURATION* Conf)
{
  DWORD Status;

  Status = IoctlSync(
              IOCTL_USBIO_SET_CONFIGURATION,
              Conf,
              sizeof(USBIO_SET_CONFIGURATION),
              NULL,
              0,
              NULL
              );

  return Status;
}

DWORD CUsbIo::UnconfigureDevice()
{
  DWORD Status;

  Status = IoctlSync(
              IOCTL_USBIO_UNCONFIGURE_DEVICE,
              NULL,
              0,
              NULL,
              0,
              NULL
              );

  return Status;
}



DWORD CUsbIo::SetInterface(const USBIO_INTERFACE_SETTING* Setting)
{
  DWORD Status;

  Status = IoctlSync(
              IOCTL_USBIO_SET_INTERFACE,
              Setting,
              sizeof(USBIO_INTERFACE_SETTING),
              NULL,
              0,
              NULL
              );

  return Status;
}



DWORD CUsbIo::ClassOrVendorInRequest(
        void* Buffer,
        DWORD& ByteCount,
        const USBIO_CLASS_OR_VENDOR_REQUEST *Request )
{
  DWORD Status;

  Status = IoctlSync(
              IOCTL_USBIO_CLASS_OR_VENDOR_IN_REQUEST,
              Request,
              sizeof(USBIO_CLASS_OR_VENDOR_REQUEST),
              Buffer,
              ByteCount,
              &ByteCount
              );

  return Status;
}



DWORD CUsbIo::ClassOrVendorOutRequest(
        const void* Buffer,
        DWORD& ByteCount,
        const USBIO_CLASS_OR_VENDOR_REQUEST *Request )
{
  DWORD Status;

  Status = IoctlSync(
              IOCTL_USBIO_CLASS_OR_VENDOR_OUT_REQUEST,
              Request,
              sizeof(USBIO_CLASS_OR_VENDOR_REQUEST),
              (void*)Buffer,
              ByteCount,
              &ByteCount
              );

  return Status;
}



DWORD CUsbIo::GetDeviceParameters(USBIO_DEVICE_PARAMETERS *DevParam)
{
  DWORD Status;

  // zero the struct if any fields are added...
  ZeroMemory(DevParam,sizeof(USBIO_DEVICE_PARAMETERS));

  Status = IoctlSync(
              IOCTL_USBIO_GET_DEVICE_PARAMETERS,
              NULL,
              0,
              DevParam,
              sizeof(USBIO_DEVICE_PARAMETERS),
              NULL
              );

  return Status;
}



DWORD CUsbIo::SetDeviceParameters(const USBIO_DEVICE_PARAMETERS *DevParam)
{
  DWORD Status;

  Status = IoctlSync(
              IOCTL_USBIO_SET_DEVICE_PARAMETERS,
              DevParam,
              sizeof(USBIO_DEVICE_PARAMETERS),
              NULL,
              0,
              NULL
              );

  return Status;
}



DWORD CUsbIo::GetConfigurationInfo(USBIO_CONFIGURATION_INFO *Info)
{
  DWORD Status;

  // zero the struct if any fields are added...
  ZeroMemory(Info,sizeof(USBIO_CONFIGURATION_INFO));

  Status = IoctlSync(
              IOCTL_USBIO_GET_CONFIGURATION_INFO,
              NULL,
              0,
              Info,
              sizeof(USBIO_CONFIGURATION_INFO),
              NULL
              );

  return Status;
}



DWORD CUsbIo::ResetDevice()
{
  DWORD Status;

  Status = IoctlSync(
              IOCTL_USBIO_RESET_DEVICE,
              NULL,
              0,
              NULL,
              0,
              NULL
              );

  return Status;
}


DWORD CUsbIo::CyclePort()
{
  DWORD Status;

  Status = IoctlSync(
              IOCTL_USBIO_CYCLE_PORT,
              NULL,
              0,
              NULL,
              0,
              NULL
              );

  return Status;
}



DWORD CUsbIo::GetCurrentFrameNumber(DWORD &FrameNumber)
{
  DWORD Status;
  USBIO_FRAME_NUMBER data;

  // zero the struct if any fields are added...
  ZeroMemory(&data,sizeof(data));

  Status = IoctlSync(
              IOCTL_USBIO_GET_CURRENT_FRAME_NUMBER,
              NULL,
              0,
              &data,
              sizeof(data),
              NULL
              );

  FrameNumber = data.FrameNumber;

  return Status;
}


DWORD CUsbIo::GetDevicePowerState(USBIO_DEVICE_POWER_STATE &DevicePowerState)
{
  USBIO_DEVICE_POWER PowerRequest;
  DWORD err;

  // zero the struct if any fields are added...
  ZeroMemory(&PowerRequest,sizeof(PowerRequest));

  err = IoctlSync(
              IOCTL_USBIO_GET_DEVICE_POWER_STATE,
              NULL,
              0,
              &PowerRequest,
              sizeof(USBIO_DEVICE_POWER),
              NULL
              );
  if ( err == USBIO_ERR_SUCCESS ) {
    DevicePowerState = PowerRequest.DevicePowerState;
  }
  return err;
}


DWORD CUsbIo::SetDevicePowerState(USBIO_DEVICE_POWER_STATE DevicePowerState)
{
  USBIO_DEVICE_POWER PowerRequest;

  PowerRequest.DevicePowerState = DevicePowerState;

  return IoctlSync(
              IOCTL_USBIO_SET_DEVICE_POWER_STATE,
              &PowerRequest,
              sizeof(USBIO_DEVICE_POWER),
              NULL,
              0,
              NULL
              );
}


DWORD CUsbIo::GetDeviceDescriptor(USB_DEVICE_DESCRIPTOR* Desc )
{
  DWORD ByteCount = sizeof(USB_DEVICE_DESCRIPTOR);

  return GetDescriptor(
              Desc,
              ByteCount,
              RecipientDevice,
              USB_DEVICE_DESCRIPTOR_TYPE,
              0,
              0
              );
}



DWORD CUsbIo::GetConfigurationDescriptor(
        USB_CONFIGURATION_DESCRIPTOR* Desc,
        DWORD& ByteCount,
        UCHAR Index/*=0*/ )
{

  return GetDescriptor(
              Desc,
              ByteCount,
              RecipientDevice,
              USB_CONFIGURATION_DESCRIPTOR_TYPE,
              Index,
              0
              );
}



DWORD CUsbIo::GetStringDescriptor(
        USB_STRING_DESCRIPTOR* Desc,
        DWORD& ByteCount,
        UCHAR Index/*=0*/,
        USHORT LanguageId/*=0*/ )
{

  return GetDescriptor(
              Desc,
              ByteCount,
              RecipientDevice,
              USB_STRING_DESCRIPTOR_TYPE,
              Index,
              LanguageId
              );
}




DWORD CUsbIo::IoctlSync(
        DWORD IoctlCode,
        const void *InBuffer,
        DWORD InBufferSize,
        void *OutBuffer,
        DWORD OutBufferSize,
        DWORD *BytesReturned
        )
{
  DWORD Status;
  DWORD BytesRet = 0;
  BOOL succ;

  // check if the driver was opened
  if ( FileHandle == NULL ) {
    return USBIO_ERR_DEVICE_NOT_OPEN;
  }

  // IOCTL requests must be serialized
  // bec. there is only one event object per instance
  EnterCriticalSection(&CritSect);

  // call the device driver
  succ = DeviceIoControl(
            FileHandle,         // driver handle
            IoctlCode,          // IOCTL code
            (void*)InBuffer,    // input buffer
            InBufferSize,       // input buffer size
            OutBuffer,          // output buffer
            OutBufferSize,      // output buffer size
            &BytesRet,          // number of bytes returned
            &Overlapped         // overlapped structure (async.)
            );
  if ( succ ) {
    // ioctl completed successfully
    Status = USBIO_ERR_SUCCESS;
  } else {
    Status = GetLastError();
    if ( Status == ERROR_IO_PENDING ) {
      // the operation is pending, wait for completion
      succ = GetOverlappedResult(
                FileHandle,
                &Overlapped,
                &BytesRet,  // byte count
                TRUE        // wait flag
                );
      if ( succ ) {
        // completed successfully
        Status = USBIO_ERR_SUCCESS;
      } else {
        Status = GetLastError();
      }
    }
  }

  LeaveCriticalSection(&CritSect);

  if ( BytesReturned != NULL ) {
    *BytesReturned = BytesRet;
  }

  return Status;
}



BOOL CUsbIo::CancelIo()
{
  // cancel all outstanding requests that were
  // issued by the calling thread on this handle
  return ::CancelIo(FileHandle);
}



// helper struct
struct _ErrorCodeTable {
  DWORD Code;
  const char *String;
};

char* CUsbIo::ErrorText(char* StringBuffer, DWORD StringBufferSize, DWORD ErrorCode)
{
  // string table
  static const struct _ErrorCodeTable ErrorTable[] = {
    {USBIO_ERR_SUCCESS                , "No error."},
    {USBIO_ERR_CRC                    , "HC Error: Wrong CRC."},
    {USBIO_ERR_BTSTUFF                , "HC Error: Wrong bitstuffing."},
    {USBIO_ERR_DATA_TOGGLE_MISMATCH   , "HC Error: Data toggle mismatch."},
    {USBIO_ERR_STALL_PID              , "HC Error: stall PID."},
    {USBIO_ERR_DEV_NOT_RESPONDING     , "HC Error: Device not responding."},
    {USBIO_ERR_PID_CHECK_FAILURE      , "HC Error: PID check failed."},
    {USBIO_ERR_UNEXPECTED_PID         , "HC Error: Unexpected PID."},
    {USBIO_ERR_DATA_OVERRUN           , "HC Error: Data Overrun."},
    {USBIO_ERR_DATA_UNDERRUN          , "HC Error: Data Underrun."},
    {USBIO_ERR_RESERVED1              , "HC Error: Reserved1."},
    {USBIO_ERR_RESERVED2              , "HC Error: Reserved2."},
    {USBIO_ERR_BUFFER_OVERRUN         , "HC Error: Buffer Overrun."},
    {USBIO_ERR_BUFFER_UNDERRUN        , "HC Error: Buffer Underrun."},
    {USBIO_ERR_NOT_ACCESSED           , "HC Error: Not accessed."},
    {USBIO_ERR_FIFO                   , "HC Error: FIFO error."},
    {USBIO_ERR_ENDPOINT_HALTED        , "USBD Error: Endpoint halted."},
    {USBIO_ERR_NO_MEMORY              , "USBD Error: No System Memory."},
    {USBIO_ERR_INVALID_URB_FUNCTION   , "USBD Error: Invalid URB function."},
    {USBIO_ERR_INVALID_PARAMETER      , "USBD Error: Invalid parameter."},
    {USBIO_ERR_ERROR_BUSY             , "USBD Error: Error: BUSY."},
    {USBIO_ERR_REQUEST_FAILED         , "USBD Error: Request failed."},
    {USBIO_ERR_INVALID_PIPE_HANDLE    , "USBD Error: Invalid pipe handle."},
    {USBIO_ERR_NO_BANDWIDTH           , "USBD Error: No bandwidth available."},
    {USBIO_ERR_INTERNAL_HC_ERROR      , "USBD Error: Internal HC error."},
    {USBIO_ERR_ERROR_SHORT_TRANSFER   , "USBD Error: Error: short transfer."},
    {USBIO_ERR_BAD_START_FRAME        , "USBD Error: Bad start frame."},
    {USBIO_ERR_ISOCH_REQUEST_FAILED   , "USBD Error: Isochronous request failed."},
    {USBIO_ERR_FRAME_CONTROL_OWNED    , "USBD Error: Frame control owned."},
    {USBIO_ERR_FRAME_CONTROL_NOT_OWNED, "USBD Error: Frame control not owned."},

    {USBIO_ERR_CANCELED               , "USBD Error: cancelled."},
    {USBIO_ERR_CANCELING              , "USBD Error: cancelling."},

    {USBIO_ERR_FAILED                 , "Operation failed."},
    {USBIO_ERR_INVALID_INBUFFER       , "Input buffer too small."},
    {USBIO_ERR_INVALID_OUTBUFFER      , "Output buffer too small."},
    {USBIO_ERR_OUT_OF_MEMORY          , "Out of memory."},
    {USBIO_ERR_PENDING_REQUESTS       , "There are pending requests. Use Abort first."},
    {USBIO_ERR_ALREADY_CONFIGURED     , "USB device is already configured."},
    {USBIO_ERR_NOT_CONFIGURED         , "USB device is not configured."},
    {USBIO_ERR_OPEN_PIPES             , "There are open pipes. Use Unbind/Close first."},
    {USBIO_ERR_ALREADY_BOUND          , "Pipe is already bound."},
    {USBIO_ERR_NOT_BOUND              , "Handle is not bound to a pipe."},
    {USBIO_ERR_DEVICE_NOT_PRESENT     , "Device is removed."},
    {USBIO_ERR_CONTROL_NOT_SUPPORTED  , "Control code is not supported."},
    {USBIO_ERR_TIMEOUT                , "Request timeout interval has expired."},
    {USBIO_ERR_INVALID_RECIPIENT      , "Invalid recipient."},
    {USBIO_ERR_INVALID_TYPE           , "Invalid pipe type. Use IOCTRL for control pipe."},
    {USBIO_ERR_INVALID_IOCTL          , "Invalid I/O control code."},
    {USBIO_ERR_INVALID_DIRECTION      , "Invalid direction of I/O operation."},
    {USBIO_ERR_TOO_MUCH_ISO_PACKETS   , "Too much ISO packets. Adjust registry parameter."},
    {USBIO_ERR_POOL_EMPTY             , "Request pool empty."},
    {USBIO_ERR_PIPE_NOT_FOUND         , "Pipe not found."},
    {USBIO_ERR_INVALID_ISO_PACKET     , "Invalid ISO packet. Offset + Length > Buffer Size!"},
    {USBIO_ERR_OUT_OF_ADDRESS_SPACE   , "Out of address space."},
    {USBIO_ERR_INTERFACE_NOT_FOUND    , "Interface not found."},
    {USBIO_ERR_INVALID_DEVICE_STATE   , "Invalid device state (stopped or power down)."},
    {USBIO_ERR_INVALID_PARAM          , "Invalid parameter."},
    {USBIO_ERR_DEMO_EXPIRED           , "DEMO version has expired! You have to reboot!"},
    {USBIO_ERR_INVALID_POWER_STATE    , "Power state not allowed. Set to D0 first."},
    {USBIO_ERR_POWER_DOWN             , "Requests cancelled while device goes power down."},
    {USBIO_ERR_VERSION_MISMATCH       , "API Version does not match."},
    {USBIO_ERR_SET_CONFIGURATION_FAILED,"Set configuration failed. Configure one interface only."},

    {USBIO_ERR_DEVICE_NOT_FOUND       , "Device not found."},
    {USBIO_ERR_DEVICE_NOT_OPEN        , "Device not open."},
    {USBIO_ERR_NO_SUCH_DEVICE_INSTANCE, "No such device instance."},
    {USBIO_ERR_INVALID_FUNCTION_PARAM,  "An invalid parameter was passed."},

    {USBIO_ERR_VID_RESTRICTION,         "Light version restriction: Unsupported Vendor ID."},
    {USBIO_ERR_ISO_RESTRICTION,         "Light version restriction: ISO pipes are not supported."},
    {USBIO_ERR_BULK_RESTRICTION,        "Light version restriction: BULK pipes are not supported."},
    {USBIO_ERR_EP0_RESTRICTION,         "Light version restriction: EP0 requests are not fully supported."},
    {USBIO_ERR_PIPE_RESTRICTION,        "Light version restriction: Too many pipes or pipe type not supported."},
    {USBIO_ERR_PIPE_SIZE_RESTRICTION,   "Light version restriction: Maximum FIFO size exceeded."}
  };

  static int Size = sizeof(ErrorTable)/sizeof(struct _ErrorCodeTable);
  const char *ErrorString = "Windows system error code.";
  int i;
//  BOOL found=FALSE;

  if ( (StringBuffer==NULL) || (StringBufferSize==0) ) {
    return StringBuffer;
  }

  for (i=0;i<Size;i++) {
    if (ErrorTable[i].Code == ErrorCode) {
      ErrorString=ErrorTable[i].String;
      //found = TRUE;
      break;
    }
  }

  // the following does not produce useful error messages
  // so we don't use it anymore
  /*
  char* MsgBuffer = NULL;
  if (!found) {
    if (0 != FormatMessage(
                FORMAT_MESSAGE_ALLOCATE_BUFFER |
                FORMAT_MESSAGE_FROM_SYSTEM | 
                FORMAT_MESSAGE_IGNORE_INSERTS,
                NULL,
                ErrorCode,
                MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
                (LPTSTR)&MsgBuffer,    
                0,    
                NULL 
                )) {
      // found
      ErrorString = MsgBuffer; 
    }
  }
  */
  // print to string buffer
  _snprintf(StringBuffer,StringBufferSize,"Error code 0x%08X: %s",ErrorCode,ErrorString); 
  // make sure the string is zero-terminated
  StringBuffer[StringBufferSize-1] = 0;

/*
  // free resources
  if ( MsgBuffer!=NULL ) {
    LocalFree(MsgBuffer);
  }
*/

  return StringBuffer;
}


/*************************** EOF **************************************/
